//
//  CoreDataHelpers.swift
//  Do It
//
//  Created by Jim Dovey on 2/14/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import CoreData
import SwiftUI

// MARK: TodoItemList

extension TodoItemList {
    var color: ListColor {
        get {
            let color = self.colorClass?.color ?? .blue
            return color
        }
        set {
            self.colorClass = ListColorClass(newValue)
        }
    }

    var incompleteCount: Int {
        guard let moc = self.managedObjectContext else {
            return 0
        }

        let request: NSFetchRequest<TodoItem> = TodoItem.fetchRequest()
        request.predicate = self.itemsPredicate

        var count = 0
        moc.performAndWait {
            count = (try? moc.count(for: request)) ?? 0
        }

        return count
    }

    private var itemsPredicate: NSPredicate {
        let left = NSExpression(forKeyPath: \TodoItem.list)
        let right = NSExpression(forConstantValue: objectID)
        return NSComparisonPredicate(
            leftExpression: left,
            rightExpression: right,
            modifier: .direct,
            type: .equalTo,
            options: [])
    }
}

extension TodoItemList {
    private static func predicateForName(_ name: String) -> NSPredicate {
        let left = NSExpression(forKeyPath: \TodoItemList.name)
        let right = NSExpression(forConstantValue: name)
        return NSComparisonPredicate(
            leftExpression: left,
            rightExpression: right,
            modifier: .direct,
            type: .equalTo,
            options: [])
    }

    private static func fetchRequest(forName name: String) -> NSFetchRequest<TodoItemList> {
        let request: NSFetchRequest<TodoItemList> = fetchRequest()
        request.predicate = predicateForName(name)
        request.resultType = .managedObjectResultType
        request.fetchLimit = 1
        request.includesPropertyValues = true
        return request
    }

    // START:AllListItemsRequest
    var requestForAllItems: NSFetchRequest<TodoItem> {
        let request: NSFetchRequest<TodoItem> = TodoItem.fetchRequest()
        request.predicate = NSPredicate(format: "list == %@", self)
        request.sortDescriptors = [
            NSSortDescriptor(keyPath: \TodoItem.manualSortOrder, ascending: true)
        ]
        request.fetchBatchSize = 25
        return request
    }
    // END:AllListItemsRequest

    static func findList(named name: String,
                         in context: NSManagedObjectContext) -> TodoItemList? {
        let request = fetchRequest(forName: name)
        var result: TodoItemList?
        context.performAndWait {
            result = try? request.execute().first
        }
        return result
    }

    static func findOrCreateList(named name: String,
                                 color: ListColor = .blue,
                                 icon: String = "list.bullet",
                                 sortOrder: Int32? = nil,
                                 in context: NSManagedObjectContext) -> TodoItemList {
        if let list = findList(named: name, in: context) {
            return list
        }

        let list = TodoItemList(context: context)
        list.name = name
        list.color = color
        list.icon = icon
        list.manualSortOrder = sortOrder ?? Int32(listCount(in: context))
        return list
    }

    static func listCount(in context: NSManagedObjectContext) -> Int {
        let request: NSFetchRequest<TodoItemList> = fetchRequest()
        return (try? context.count(for: request)) ?? 0
    }

    // START:NewList
    static func newList(in context: NSManagedObjectContext) -> TodoItemList {
        let list = TodoItemList(context: context)
        list.name = NSLocalizedString("New List", comment: "Default title for new lists")
        list.icon = "list.bullet"
        list.color = .blue
        list.manualSortOrder = Int32(listCount(in: context))
        return list
    }
    // END:NewList

    static func defaultList(in context: NSManagedObjectContext) -> TodoItemList {
        let request: NSFetchRequest<TodoItemList> = fetchRequest()
        request.predicate = NSComparisonPredicate(
            leftExpression: NSExpression(forKeyPath: \TodoItemList.isDefaultList),
            rightExpression: NSExpression(forConstantValue: true),
            modifier: .direct,
            type: .equalTo,
            options: [])
        if let result = try? context.fetch(request), !result.isEmpty {
            return result[0]
        }
        else {
            // Create a new default list
            let list = newList(in: context)
            list.name = "Default list"
            try? context.save()
            return list
        }
    }
    
    static func allLists(in context: NSManagedObjectContext) -> [TodoItemList] {
        let request: NSFetchRequest<TodoItemList> = fetchRequest()
        if let result = try? context.fetch(request) {
            return result
        }
        else {
            return []
        }
    }
}

// MARK: - TodoItem

extension TodoItem {
    var priority: Priority {
        get {
            switch rawPriority {
            case 0: return .low
            case 1: return .normal
            case 2: return .high
            case 3: return .urgent
            default:
                rawPriority = 1
                return .normal
            }
        }
        set {
            switch newValue {
            case .low: rawPriority = 0
            case .normal: rawPriority = 1
            case .high: rawPriority = 2
            case .urgent: rawPriority = 3
            }
        }
    }

    var complete: Bool {
        get { self.completed != nil }
        set {
            if newValue {
                self.completed = Date()
            }
            else {
                self.completed = nil
            }
        }
    }
}

extension TodoItem {
    var usesTimeOfDay: Bool {
        get {
            guard let date = self.date else { return false }
            return !Calendar.current
                .date(date, matchesComponents:
                    DateComponents(hour: 0, minute: 0, second: 0))
        }
        set {
            guard let date = self.date else {
                if newValue {
                    // we can't set this to true and not have a date afterward;
                    // if that were the case, you could have x = true; x == false
                    self.date = Date()
                }
                return
            }

            if (newValue) {
                if Calendar.current.isDateInToday(date) {
                    self.date = Date() // current time
                }
                else {
                    // use midday of whichever day is currently indicated
                    self.date = Calendar.current.date(bySettingHour: 12,
                                                      minute: 0,
                                                      second: 0,
                                                      of: date) ?? Date()
                }
            }
            else {
                // set to midnight on the target date.
                self.date = Calendar.current.date(bySettingHour: 0,
                                                  minute: 0,
                                                  second: 0,
                                                  of: date)
            }
        }
    }
}

extension TodoItem {
    static var notCompletePredicate: NSPredicate {
        let left = NSExpression(forKeyPath: \TodoItem.completed)
        let greater = NSComparisonPredicate(
            leftExpression: left,
            rightExpression: NSExpression(forConstantValue: Date() as NSDate),
            modifier: .direct,
            type: .greaterThan,
            options: [])

        let isNil = NSComparisonPredicate(
            leftExpression: left,
            rightExpression: NSExpression(forConstantValue: nil),
            modifier: .direct,
            type: .equalTo,
            options: [])

        return NSCompoundPredicate(orPredicateWithSubpredicates: [isNil, greater])
    }

    static func newTodoItem(in list: TodoItemList) -> TodoItem {
        let context = list.managedObjectContext!
        let count = try? context.count(for: fetchRequest())

        let item = TodoItem(context: context)
        item.title = NSLocalizedString("New Item", comment: "Default to-do title")
        item.priority = .normal
        item.manualSortOrder = Int32(count ?? .max)
        item.list = list
        return item
    }
}

// MARK: - Managed Object

extension NSManagedObject: Identifiable {
    public var id: NSManagedObjectID { objectID }
}

extension NSManagedObject {
    func revert() {
        guard let moc = self.managedObjectContext else { return }
        if self.isInserted {
            moc.delete(self)
        }
        else if self.hasPersistentChangedValues {
            moc.refresh(self, mergeChanges: false)
        }
    }
}

// MARK: - Object Context

extension NSManagedObjectContext {
    func editingContext() -> NSManagedObjectContext {
        let context = NSManagedObjectContext(concurrencyType: .mainQueueConcurrencyType)
        context.parent = self
        return context
    }
    
    func realize<T: NSManagedObject>(_ obj: T) -> T? {
        try? existingObject(with: obj.objectID) as? T
    }
}

// MARK: - SwiftUI FetchRequest

extension FetchRequest where Result: NSManagedObject {
    init<Value>(ascending keyPath: KeyPath<Result, Value>) {
        self.init(sortDescriptors: [NSSortDescriptor(keyPath: keyPath, ascending: true)])
    }
}
